import platform    # For getting the operating system name
import subprocess  # For executing a shell command
import threading
import time
import re
import json

def run(command):
    pipe = subprocess.Popen(command,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.STDOUT)
    stdout, stderr = pipe.communicate()
    status = pipe.returncode
    return (status, stdout)

def ping(host, results):
    """
    Returns True if host (str) responds to a ping request.
    Remember that a host may not respond to a ping (ICMP) request even if the host name is valid.
    """

    # Option for the number of packets as a function of
    param = '-n' if platform.system().lower()=='windows' else '-c'

    # Option for the number of packets as a function of
    paramw1 = '-w' if platform.system().lower()=='windows' else '-W'
    paramw2 = '100' if platform.system().lower()=='windows' else '1'

    # Building the command. Ex: "ping -c 1 google.com"
    command = ['ping', param, '1', paramw1, paramw2, host]

    (rc, output) = run(command)

    up = False
    if rc == 0:
        status = "Reachable"
        up = True
    else:
        status = "Not reachable"

    results[host] = {'ip': host, 'up': up, 'status': status}
    return


def sendPing(addr, results):
    server_thread = threading.Thread(target=ping, args=(addr, results))
    server_thread.daemon = True
    server_thread.start()
#    print('ping thread launched')

def getMACs(results):
    (status, output) = run(['arp', '-a'])

    lines = output.decode('ascii').split('\n')

    lines = map(lambda s: s.rstrip(), lines)

    rexp = ' +([0-9.]+) +([0-9a-f-]+).*'

    lines = map(lambda s: re.match(rexp, s), lines)

    lines = map(lambda s: {'ip': s.group(1), 'mac': s.group(2)} if s is not None else {}, lines)

    res = [f for f in lines if len(f.keys())]
    return res

def sendPings():
    results = {}
    for i in range(253):
        sendPing("192.168.2.%d" % (1+i,), results)

    time.sleep(1)

    live = { f: results[f] for f in results if results[f]['up'] }

    maclist = getMACs(live)
    macdict = { maclist[k]['ip']: maclist[k] for k in range(len(maclist)) }

    for f in live:
        ent = live[f]
        try:
            ent['mac'] = macdict[ent['ip']]['mac']
        except:
            ent['mac'] = '???'

    return live

def filterByMAC(live, MACprefix):
    return {l: live[l] for l in live if live[l]['mac'].startswith(MACprefix)}

def filterByHostname(live, name):
    return {l: live[l] for l in live if name in live[l]['hostname']}

def getHostnames(live):
    for l in live:
        ip = live[l]['ip']
        rc, res = run(['nslookup', ip])
        res = res.decode('ascii')
        lines = res.split('\n')
        for k in range(len(lines)):
            s = re.match('Name:', lines[k])
            if s is not None:
                live[l]['hostname'] = lines[k][6:].strip()
    return live

raspiMACPrefix = 'b8-27'
usenseHostnamePrefix = 'usense'

if __name__ == "__main__":
    live = sendPings()
    live = filterByHostname(getHostnames(filterByMAC(live, raspiMACPrefix)), 'usense')
    print(json.dumps(live, indent=2))
